import os
import json
import requests
import logging
from datetime import datetime
import tkinter as tk
from tkinter import ttk, messagebox, scrolledtext

# ==========================
# LOGS
# ==========================
LOG_FILE = os.path.join(os.getcwd(), "pedido_log.txt")
logging.basicConfig(
    level=logging.INFO,
    format="%(asctime)s [%(levelname)s] %(message)s",
    handlers=[logging.FileHandler(LOG_FILE, mode="a", encoding="utf-8")]
)
logging.info("=== Inicio de ejecución ===")

# ==========================
# CONFIGURACIÓN Canales 
# ==========================
CHANNELS = {
    "tnannastein": ["pickup", "delivery"],
    "magento": ["pickup", "delivery"],
    "MELI": ["pickup", "delivery"],
    "farmaciadigital": ["pickup", "delivery"],
    "farmatouch": ["pickup", "delivery"],
    "RAPPI": ["delivery"],
    "PEYA": ["delivery"],
    "benvida": ["pickup", "delivery"],
    "vtex": ["delivery"],
    "SF": ["delivery"]
}

EXTERNAL_FILE = "external_number.txt"
INITIAL_EXTERNAL = 440000000   ##EXTERNAL NUMBER QUE VAS A UTILIZAR
SHIPPING_COST = 15
PROMO_DISCOUNT = 2400

URLS_CLIENTES = {
    "Farmacia Central Oeste - QA": "https://bmc-foco.dev.napse.global:8444/order/create",
    "Proyecto 2": "https://dev.3napse.global:8444/order/create",
    "Proyecto 3": "https://napse.4global:8444/order/create"
}

# ==========================
# FUNCIONES
# ==========================
def get_next_external_number():
    if os.path.exists(EXTERNAL_FILE):
        with open(EXTERNAL_FILE, "r") as f:
            num = int(f.read().strip())
    else:
        num = INITIAL_EXTERNAL - 1
    next_num = num + 1
    with open(EXTERNAL_FILE, "w") as f:
        f.write(str(next_num))
    logging.info(f"ExternalNumber generado: {next_num}")
    return str(next_num)

def generar_pedido():
    cliente_seleccionado = cliente_var.get()
    URL = URLS_CLIENTES[cliente_seleccionado]
    channelCode = canal_var.get()
    deliveryOrPickup = entrega_var.get()
    storeCode = store_var.get()
    isPaid = pagado_var.get()
    aplicaPromo = promo_var.get()
    aplica_descuento_manual = descuento_manual_var.get()

    creationDate = datetime.now().strftime("%Y-%m-%d")
    externalNumber = get_next_external_number()

    # Datos fijos
    pedido = {
        "externalNumber": externalNumber,
        "creationDate": creationDate,
        "channelCode": channelCode,
        "currencyCode": "ARS",
        "manualDiscountAmount": 0,
        "netAmount": 11000,
        "orderTypeCode": "web",
        "isPaid": isPaid,
        "party": {
            "code": "",
            "firstName": "Alejandro",
            "lastName": "Isola",
            "email": "Leo@napse.global",
            "typeCode": "Person",
            "identificationType": 1,
            "identifier": "2995028",
            "partyContactMethods": [
                {
                    "name": "personal",
                    "address": {
                        "firstLine": "Av San Aurelio 7844",
                        "countryCode": "AR",
                        "stateCode": "BA",
                        "cityCode": "BSAS_194"
                    },
                    "emailAddress": "alejandro.silva@napse.global",
                    "telephone": {
                        "telephoneNumber": "0342 4744900"
                    },
                }
            ]
        },
        "detail": [
            {
                "itemCode": "1477",
                "description": "BRAVA DESOD. LUBRICANTE-sachet cajax20 cód.12060",
                "storeCode": storeCode,
                "locationCode": "DEP1",
                "unitPrice": 5000,
                "orderedQty": 2,
                "deliveryOrPickup": deliveryOrPickup,
                "deliveryCompany": "100000003",
                "pickupLocationCode": "100000003",
                "packageId": "0",
                "partyContactMethodName": "personal",
                "pickupInformation": "Info: Alejandro Silva Mirgor retira por tienda."
            },
            {
                "itemCode": "9018",
                "description": "BETAVALMAX-60 mg gts.x 15 ml",
                "storeCode": storeCode,
                "locationCode": "DEP1",
                "unitPrice": 1000,
                "orderedQty": 1,
                "deliveryOrPickup": deliveryOrPickup,
                "deliveryCompany": "100000003",
                "pickupLocationCode": "100000003",
                "packageId": "0",
                "partyContactMethodName": "personal",
                "pickupInformation": "Info: Alejandro Silva Mirgor retira por tienda."
            }
        ]
    }
# ===================================
# Detalles de delivery y su importe
# =================================== 

    if deliveryOrPickup == "delivery":
        envio_item = {
            "itemCode": "ENVIO",
            "description": "Costo de Envio",
            "storeCode": storeCode,
            "locationCode": "DEP1",
            "unitPrice": SHIPPING_COST,
            "orderedQty": 1,
            "deliveryOrPickup": "delivery",
            "deliveryCompany": "magento",
            "pickupLocationCode": "100000003",
            "packageId": "0",
            "partyContactMethodName": "personal",
            "pickupInformation": "Info: Alejandro Silva por delivery.",
            "notes": "RECETA=true|CODIGO=889999|MATMED=99999988888",
            "sellerID": "omni",
            "sellerName": "Usuario para WF OMNI"
        }
        pedido["detail"].append(envio_item)
        pedido["netAmount"] += SHIPPING_COST

    if aplica_descuento_manual:
        try:
            descuento = float(descuento_entry.get())
            if 0 < descuento <= pedido["netAmount"]:
                pedido["manualDiscountAmount"] = descuento
                pedido["netAmount"] -= descuento
            else:
                messagebox.showwarning("Advertencia", "Descuento inválido.")
        except ValueError:
            messagebox.showwarning("Advertencia", "Debe ingresar un número válido para el descuento.")

    if aplicaPromo:
        pedido["detail"][0]["priceModifiers"] = [
            {"percent": 20, "amount": PROMO_DISCOUNT, "reasonText": "38"}
        ]
        pedido["netAmount"] -= PROMO_DISCOUNT

    if isPaid:
        pago = {
            "tender": "EL",
            "amount": pedido["netAmount"],
            "currencyCode": "ARS",
            "planDescriptor": "12",
            "sellerName": "vISA CREDITO",
            "authorizationCode": "9187840193",
            "couponNumber": "123",
            "lotNumber": "5321",
            "installments": 6,
            "referenceNumber": "583409####0415"
        }
        pedido["payments"] = [pago]
    else:
        pedido["payments"] = []

    # Mostrar pedido en ventana
    preview_text.config(state="normal")
    preview_text.delete("1.0", tk.END)
    preview_text.insert(tk.END, json.dumps(pedido, indent=4, ensure_ascii=False))
    preview_text.config(state="disabled")

    # Guardar pedido para envío
    app.pedido_actual = pedido
    app.URL = URL

def enviar_pedido():
    if not hasattr(app, 'pedido_actual'):
        messagebox.showwarning("Advertencia", "Primero genere el pedido.")
        return
    token = token_entry.get().strip()
    if not token:
        messagebox.showwarning("Advertencia", "Ingrese el x-access-token.")
        return
    headers = {"Content-Type": "application/json", "x-access-token": token}
    try:
        response = requests.post(app.URL, headers=headers, json=app.pedido_actual)
        messagebox.showinfo("Resultado", f"Status: {response.status_code}\nRespuesta: {response.text}")
        logging.info(f"Pedido enviado a {app.URL}")
        logging.info(f"Respuesta: {response.text}")
    except Exception as e:
        messagebox.showerror("Error", f"No se pudo enviar: {e}")
        logging.error(f"Error al enviar: {e}")

# ==========================
# INTERFAZ GRÁFICA CON COLORES
# ==========================
COLOR_FONDO = "#2c004b"   # Fondo general
COLOR_TEXTO = "#d27960"   # Texto
COLOR_BOTON = "#ffb200"   # Botones

# ==========================
# INTERFAZ GRÁFICA CON TÍTULO Y CENTRADO
# ==========================
app = tk.Tk()
app.title("Generador de Pedido Omni (QA)")
app.geometry("900x700")
app.configure(bg=COLOR_FONDO)

# Configuramos las columnas para que se expandan y centren
for i in range(3):
    app.grid_columnconfigure(i, weight=1)

# Título centrado
titulo = tk.Label(app, text="Generador de Pedidos Omni (Ambiente QA)", fg=COLOR_TEXTO, bg=COLOR_FONDO, font=("Arial", 20, "bold"))
titulo.grid(row=0, column=0, columnspan=3, pady=15)

# Función para crear labels centrados
def crear_label(texto, fila):
    tk.Label(app, text=texto, fg=COLOR_TEXTO, bg=COLOR_FONDO, font=("Arial", 12)).grid(row=fila, column=0, columnspan=3, sticky="ew", pady=2)

# Cliente
crear_label("Cliente:", 1)
cliente_var = tk.StringVar()
cliente_menu = ttk.Combobox(app, textvariable=cliente_var, values=list(URLS_CLIENTES.keys()), width=50)
cliente_menu.grid(row=2, column=0, columnspan=3)
cliente_menu.current(0)

# Canal
crear_label("Canal:", 3)
canal_var = tk.StringVar()
canal_menu = ttk.Combobox(app, textvariable=canal_var, values=list(CHANNELS.keys()), width=50)
canal_menu.grid(row=4, column=0, columnspan=3)
canal_menu.current(0)

# Tipo de entrega
crear_label("Tipo de entrega:", 5)
entrega_var = tk.StringVar(value="pickup")
ttk.Radiobutton(app, text="pickup", variable=entrega_var, value="pickup").grid(row=6, column=0,columnspan=2)
ttk.Radiobutton(app, text="delivery", variable=entrega_var, value="delivery").grid(row=6, column=1,columnspan=3)

# Tienda
crear_label("Tienda:", 7)
store_var = tk.StringVar()
store_menu = ttk.Combobox(app, textvariable=store_var, values=["100000001","100000002","100000003","100000004","100000005","100000006","100000007","100000008"], width=50)
store_menu.grid(row=8, column=0, columnspan=3)
store_menu.current(0)

# Pagado
pagado_var = tk.BooleanVar()
tk.Checkbutton(app, text="Está pagado", variable=pagado_var, fg=COLOR_TEXTO, bg=COLOR_FONDO, selectcolor=COLOR_FONDO).grid(row=9, column=0, columnspan=3)

# Promo
promo_var = tk.BooleanVar()
tk.Checkbutton(app, text="Aplicar PROMO", variable=promo_var, fg=COLOR_TEXTO, bg=COLOR_FONDO, selectcolor=COLOR_FONDO).grid(row=10, column=0, columnspan=3)

# Descuento manual
descuento_manual_var = tk.BooleanVar()
def toggle_descuento():
    if descuento_manual_var.get():
        descuento_entry.grid(row=12, column=0, columnspan=3)
    else:
        descuento_entry.grid_forget()
tk.Checkbutton(app, text="Aplicar descuento manual", variable=descuento_manual_var, command=toggle_descuento,
               fg=COLOR_TEXTO, bg=COLOR_FONDO, selectcolor=COLOR_FONDO).grid(row=11, column=0, columnspan=3)
descuento_entry = tk.Entry(app, bg="#4b0070", fg=COLOR_TEXTO, insertbackground=COLOR_TEXTO)

# Botones
tk.Button(app, text="Generar Pedido", command=generar_pedido, bg=COLOR_BOTON, fg=COLOR_FONDO, width=20).grid(row=13, column=0, columnspan=2, pady=10)
tk.Button(app, text="Enviar Pedido", command=enviar_pedido, bg=COLOR_BOTON, fg=COLOR_FONDO, width=20).grid(row=13, column=1, columnspan=3, pady=5)

# Token
crear_label("x-access-token:", 15)
token_entry = tk.Entry(app, width=50, bg="#4b0070", fg=COLOR_TEXTO, insertbackground=COLOR_TEXTO)
token_entry.grid(row=16, column=0, columnspan=3)

# Vista previa del pedido
crear_label("Vista previa del pedido:", 17)
preview_text = scrolledtext.ScrolledText(app, width=100, height=20, bg="#3e006e", fg=COLOR_TEXTO, insertbackground=COLOR_TEXTO)
preview_text.grid(row=18, column=0, columnspan=3)
preview_text.config(state="disabled")

# Firma registrada al final
firma = tk.Label(app, text="Desarrollado por Alejandro Silva", 
                 bg=COLOR_FONDO, fg=COLOR_TEXTO, 
                 font=("Arial", 9, "italic"))
firma.grid(row=19, column=0, columnspan=3, pady=10)  # Centrado en toda la fila
app.mainloop()
